﻿namespace Microsoft.Samples.PlanMyNight.AddIns.PrintItinerary.Tests.Controllers
{
    using System;
    using System.Linq;
    using System.Security.Principal;
    using System.Web.Mvc;
    using Microsoft.Samples.PlanMyNight.AddIns.PrintItinerary.Controllers;
    using Microsoft.Samples.PlanMyNight.Data;
    using Microsoft.Samples.PlanMyNight.Entities;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using Moq;
    using MvcContrib.TestHelper;

    [TestClass]
    public class PrintItineraryControllerFixture
    {
        [TestMethod]
        public void ImportingConstructorCallServiceFactory()
        {
            var serviceFactory = new Mock<IServiceFactory>();
            serviceFactory.Setup(o => o.GetItineraryContainerInstance()).Returns(new Mock<IItineraryContainer>().Object);
            serviceFactory.Setup(o => o.GetItinerariesRepositoryInstance()).Returns(new Mock<IItinerariesRepository>().Object);
            serviceFactory.Setup(o => o.GetActivitiesRepositoryInstance()).Returns(new Mock<IActivitiesRepository>().Object);

            new PrintItineraryController(serviceFactory.Object);
            serviceFactory.VerifyAll();
        }

        [TestMethod]
        public void PrintCurrentItineraryReturnsActivitiesInCorrectOrder()
        {
            var itinerary = new Itinerary() { Name = "Night Out" };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "10", Order = 2 });

            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);

            var activitiesRepository = new Mock<IActivitiesRepository>();
            activitiesRepository.Setup(o => o.PopulateItineraryActivities(It.IsAny<Itinerary>()))
                                .Callback((Itinerary i) =>
                                {
                                    i.Activities.Where(a => a.ActivityId == "1").SingleOrDefault().Activity = new Activity { Id = "1", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                    i.Activities.Where(a => a.ActivityId == "6").SingleOrDefault().Activity = new Activity { Id = "6", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                    i.Activities.Where(a => a.ActivityId == "10").SingleOrDefault().Activity = new Activity { Id = "10", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                });

            var builder = new TestControllerBuilder();
            var controller = new PrintItineraryController(container.Object, new Mock<IItinerariesRepository>().Object, activitiesRepository.Object);
            builder.InitializeController(controller);

            var result = controller.Print(null);
            var view = result.AssertResultIs<ViewResult>();
            var model = view.ViewData.Model as Itinerary;

            StringAssert.Contains(view.ViewName, "Print");
            Assert.IsNotNull(model);
            Assert.AreEqual("Night Out", model.Name);

            // verify order
            Assert.AreEqual(3, model.Activities.Count);
            Assert.AreEqual(1, model.Activities[0].Order);
            Assert.AreEqual(2, model.Activities[1].Order);
            Assert.AreEqual(3, model.Activities[2].Order);

            activitiesRepository.VerifyAll();
        }

        [TestMethod]
        public void PrintCurrentEmptyOrNullItineraryReturnsEmptyResult()
        {
            var itinerary = new Itinerary() { Name = "An empty itinerary" };

            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);

            var builder = new TestControllerBuilder();
            var controller = new PrintItineraryController(container.Object, new Mock<IItinerariesRepository>().Object, new Mock<IActivitiesRepository>().Object);
            builder.InitializeController(controller);

            var result = controller.Print(null);
            var view = result.AssertResultIs<EmptyResult>();
        }

        [TestMethod]
        public void PrintSpecificItineraryReturnsItineraryFromRepository()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary = new Itinerary() { Name = "Night Out", Id = 15L, UserId = new Guid(UserId) };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "10", Order = 2 });

            var activitiesRepository = new Mock<IActivitiesRepository>();
            activitiesRepository.Setup(o => o.PopulateItineraryActivities(It.IsAny<Itinerary>()))
                                .Callback((Itinerary i) =>
                                {
                                    i.Activities.Where(a => a.ActivityId == "1").SingleOrDefault().Activity = new Activity { Id = "1", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                    i.Activities.Where(a => a.ActivityId == "6").SingleOrDefault().Activity = new Activity { Id = "6", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                    i.Activities.Where(a => a.ActivityId == "10").SingleOrDefault().Activity = new Activity { Id = "10", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                });

            var repository = new Mock<IItinerariesRepository>();
            repository.Setup(o => o.Retrieve(15L)).Returns(itinerary);

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new PrintItineraryController(new Mock<IItineraryContainer>().Object, repository.Object, activitiesRepository.Object);
            builder.InitializeController(controller);

            var result = controller.Print(15L);
            var view = result.AssertResultIs<ViewResult>();
            var model = view.ViewData.Model as Itinerary;

            StringAssert.Contains(view.ViewName, "Print");
            Assert.IsNotNull(model);
            Assert.AreEqual("Night Out", model.Name);
            activitiesRepository.VerifyAll();
        }

        [TestMethod]
        public void PrintSpecificItineraryForDifferentUserIsNotAllowed()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary = new Itinerary() { Name = "Night Out", Id = 15L, UserId = Guid.NewGuid() };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "10", Order = 2 });

            var repository = new Mock<IItinerariesRepository>();
            repository.Setup(o => o.Retrieve(15L)).Returns(itinerary);

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new PrintItineraryController(new Mock<IItineraryContainer>().Object, repository.Object, new Mock<IActivitiesRepository>().Object);
            builder.InitializeController(controller);

            var result = controller.Print(15L);
            var view = result.AssertResultIs<EmptyResult>();
        }

        [TestMethod]
        public void PrintSpecificItineraryForDifferentUserIfPublic()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary = new Itinerary() { Name = "Night Out", Id = 15L, IsPublic = true, UserId = Guid.NewGuid() };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "10", Order = 2 });

            var repository = new Mock<IItinerariesRepository>();
            repository.Setup(o => o.Retrieve(15L)).Returns(itinerary);

            var activitiesRepository = new Mock<IActivitiesRepository>();
            activitiesRepository.Setup(o => o.PopulateItineraryActivities(It.IsAny<Itinerary>()))
                                .Callback((Itinerary i) =>
                                {
                                    i.Activities.Where(a => a.ActivityId == "1").SingleOrDefault().Activity = new Activity { Id = "1", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                    i.Activities.Where(a => a.ActivityId == "6").SingleOrDefault().Activity = new Activity { Id = "6", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                    i.Activities.Where(a => a.ActivityId == "10").SingleOrDefault().Activity = new Activity { Id = "10", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                });

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new PrintItineraryController(new Mock<IItineraryContainer>().Object, repository.Object, activitiesRepository.Object);
            builder.InitializeController(controller);

            var result = controller.Print(15L);
            var view = result.AssertResultIs<ViewResult>();
            var model = view.ViewData.Model as Itinerary;

            StringAssert.Contains(view.ViewName, "Print");
            Assert.IsNotNull(model);
            Assert.AreEqual("Night Out", model.Name);
            activitiesRepository.VerifyAll();
        }
    }
}
